
from datetime import datetime
from typing import Optional
from pydantic import BaseModel, Field
from typing import Optional
from datetime import datetime
from enum import Enum
class NotificationOut(BaseModel):
    id: str
    title: str
    message: str
    type: str
    priority: str
    is_read: bool
    action_url: Optional[str]
    amount: Optional[float]
    merchant: Optional[str]
    created_at: datetime
    read_at: Optional[datetime]

    class Config:
        orm_mode = True


# ===== ENUMS =====
class NotificationPriority(str, Enum):
    LOW = "low"
    MEDIUM = "medium"
    HIGH = "high"
    URGENT = "urgent"

class NotificationType(str, Enum):
    NFC_PAYMENT = "nfc_payment"
    LOAN_DUE = "loan_due"
    PAYMENT = "payment"
    WITHDRAWAL = "withdrawal"
    ALERT = "alert"
    SUCCESS = "success"
    INFO = "info"

# ===== SCHEMA DE BASE =====
class NotificationBase(BaseModel):
    title: str = Field(..., min_length=1, max_length=255)
    message: str = Field(..., min_length=1)
    type: str = Field(..., description="Type de notification")
    priority: Optional[NotificationPriority] = NotificationPriority.MEDIUM
    action_url: Optional[str] = None
    amount: Optional[float] = None
    merchant: Optional[str] = None

# ===== SCHEMA POUR CRÉATION =====
class NotificationCreate(NotificationBase):
    user_id: str = Field(..., description="ID de l'utilisateur destinataire")

    class Config:
        json_schema_extra = {
            "example": {
                "user_id": "user_123",
                "title": "Nouveau paiement NFC",
                "message": "Vous avez effectué un paiement de 5000 FCFA chez Carrefour",
                "type": "nfc_payment",
                "priority": "medium",
                "action_url": "/transactions/abc123",
                "amount": 5000.0,
                "merchant": "Carrefour Market"
            }
        }

# ===== SCHEMA POUR MISE À JOUR =====
class NotificationUpdate(BaseModel):
    is_read: Optional[bool] = Field(None, description="Statut de lecture")

    class Config:
        json_schema_extra = {
            "example": {
                "is_read": True
            }
        }

# ===== SCHEMA POUR RÉPONSE (OUTPUT) =====
class NotificationOut(BaseModel):
    id: str
    user_id: str
    title: str
    message: str
    type: str
    priority: NotificationPriority
    is_read: bool
    action_url: Optional[str] = None
    amount: Optional[float] = None
    merchant: Optional[str] = None
    created_at: datetime
    read_at: Optional[datetime] = None

    class Config:
        from_attributes = True  # Pour SQLAlchemy 2.0+ (ancien orm_mode = True)
        json_schema_extra = {
            "example": {
                "id": "notif_abc123",
                "user_id": "user_123",
                "title": "Nouveau paiement NFC",
                "message": "Vous avez effectué un paiement de 5000 FCFA",
                "type": "nfc_payment",
                "priority": "medium",
                "is_read": False,
                "action_url": "/transactions/abc123",
                "amount": 5000.0,
                "merchant": "Carrefour Market",
                "created_at": "2025-10-28T10:30:00Z",
                "read_at": None
            }
        }

# ===== SCHEMA POUR LISTE DE NOTIFICATIONS =====
class NotificationList(BaseModel):
    notifications: list[NotificationOut]
    total: int
    unread_count: int

    class Config:
        json_schema_extra = {
            "example": {
                "notifications": [],
                "total": 25,
                "unread_count": 5
            }
        }

# ===== SCHEMA POUR MARQUAGE EN MASSE =====
class BulkMarkReadRequest(BaseModel):
    notification_ids: list[str] = Field(..., min_length=1, description="Liste des IDs à marquer")

    class Config:
        json_schema_extra = {
            "example": {
                "notification_ids": ["notif_1", "notif_2", "notif_3"]
            }
        }

# ===== SCHEMA POUR RÉPONSE D'ACTION EN MASSE =====
class BulkActionResponse(BaseModel):
    message: str
    count: int

    class Config:
        json_schema_extra = {
            "example": {
                "message": "5 notification(s) marquée(s) comme lue(s)",
                "count": 5
            }
        }

# ===== SCHEMA POUR COMPTEUR NON LU =====
class UnreadCountResponse(BaseModel):
    unread_count: int

    class Config:
        json_schema_extra = {
            "example": {
                "unread_count": 12
            }
        }

# ===== SCHEMA POUR SUPPRESSION =====
class DeleteResponse(BaseModel):
    message: str

    class Config:
        json_schema_extra = {
            "example": {
                "message": "Notification supprimée avec succès"
            }
        }

# ===== SCHEMA POUR FILTRES (OPTIONNEL) =====
class NotificationFilters(BaseModel):
    unread_only: Optional[bool] = False
    type: Optional[str] = None
    priority: Optional[NotificationPriority] = None
    start_date: Optional[datetime] = None
    end_date: Optional[datetime] = None
    limit: Optional[int] = Field(50, ge=1, le=100)
    offset: Optional[int] = Field(0, ge=0)

    class Config:
        json_schema_extra = {
            "example": {
                "unread_only": True,
                "type": "nfc_payment",
                "priority": "high",
                "limit": 20,
                "offset": 0
            }
        }