#!/usr/bin/env python3
"""
Nettoyer automatiquement les données hardcodées critiques
"""

import os
import shutil
from pathlib import Path

def backup_file(filepath):
    """Créer une sauvegarde avant modification"""
    backup_path = f"{filepath}.backup"
    shutil.copy2(filepath, backup_path)
    return backup_path

def cleanup_cinetpay_keys():
    """Nettoyer les clés CinetPay hardcodées"""
    
    print("\n🔑 Nettoyage des clés API CinetPay...")
    
    file_path = "utility/transactions.py"
    
    if not os.path.exists(file_path):
        print(f"   ⚠️  {file_path} non trouvé")
        return
    
    with open(file_path, 'r', encoding='utf-8') as f:
        content = f.read()
    
    # Remplacer la clé hardcodée par variable d'environnement
    original_content = content
    
    # Remplacer les clés hardcodées
    content = content.replace(
        'CINETPAY_API_KEY = "27910410466fda66bbc6a87.93656172"',
        'CINETPAY_API_KEY = os.getenv("CINETPAY_API_KEY", "")'
    )
    
    content = content.replace(
        'CINETPAY_SITE_ID = "5886416"',
        'CINETPAY_SITE_ID = os.getenv("CINETPAY_SITE_ID", "")'
    )
    
    content = content.replace(
        'CINETPAY_SECRET_KEY = "eLYyNnlIUAl8GYMQCvH5tNuQJIvNb0FmKnmNsQ6S"',
        'CINETPAY_SECRET_KEY = os.getenv("CINETPAY_SECRET_KEY", "")'
    )
    
    # Ajouter l'import os si absent
    if 'import os' not in content and 'os.getenv' in content:
        lines = content.split('\n')
        # Insérer après les autres imports
        for i, line in enumerate(lines):
            if line.startswith('import ') or line.startswith('from '):
                lines.insert(i, 'import os')
                break
        content = '\n'.join(lines)
    
    if content != original_content:
        backup_file(file_path)
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(content)
        print(f"   ✅ {file_path} nettoyé (clés API → variables d'environnement)")
    else:
        print(f"   ⚪ {file_path} déjà propre")

def cleanup_hardcoded_urls():
    """Nettoyer les URLs hardcodées"""
    
    print("\n🔗 Nettoyage des URLs hardcodées...")
    
    files_to_check = [
        "services/users.py",
        "utility/send_email.py",
    ]
    
    for file_path in files_to_check:
        if not os.path.exists(file_path):
            continue
        
        with open(file_path, 'r', encoding='utf-8') as f:
            content = f.read()
        
        original_content = content
        
        # Remplacer l'URL hardcodée du logo
        content = content.replace(
            'https://yourdomain.com/logo.png',
            '{os.getenv("APP_BASE_URL", "https://timcash.ci")}/logo.png'
        )
        
        # Remplacer l'URL des uploads
        content = content.replace(
            'http://192.168.1.87:8001/uploads/',
            '{os.getenv("APP_BASE_URL", "http://localhost:8001")}/uploads/'
        )
        
        if content != original_content:
            backup_file(file_path)
            with open(file_path, 'w', encoding='utf-8') as f:
                f.write(content)
            print(f"   ✅ {file_path} nettoyé")

def remove_test_data_files():
    """Supprimer les fichiers contenant des données de test hardcodées"""
    
    print("\n🗑️  Suppression des fichiers de données de test...")
    
    test_files = [
        "create_extended_database.py",  # Contient 52 emails de test
        "seed_data.py",  # Données de seed hardcodées
        "recreate_database.py",  # Données hardcodées
        "create_mysql_extended.py",  # Données de test
    ]
    
    for file_path in test_files:
        if os.path.exists(file_path):
            # Créer un backup
            backup_dir = "backups_hardcoded"
            os.makedirs(backup_dir, exist_ok=True)
            backup_path = os.path.join(backup_dir, os.path.basename(file_path))
            shutil.copy2(file_path, backup_path)
            
            # Supprimer le fichier original
            os.remove(file_path)
            print(f"   ✅ {file_path} supprimé (backup dans {backup_dir})")
        else:
            print(f"   ⚪ {file_path} déjà supprimé")

def update_env_example():
    """Mettre à jour .env.example avec toutes les variables nécessaires"""
    
    print("\n📝 Mise à jour de .env.example...")
    
    env_example_content = """# Configuration TIM CASH Backend - TEMPLATE

# ====================================
# BASE DE DONNÉES
# ====================================
DATABASE_URL=mysql+pymysql://timcash:timcash123@localhost/timcash_db

# MySQL Configuration
MYSQL_HOST=localhost
MYSQL_PORT=3306
MYSQL_USER=timcash
MYSQL_PASSWORD=your-secure-mysql-password
MYSQL_DATABASE=timcash_db

# ====================================
# JWT & SÉCURITÉ
# ====================================
JWT_SECRET_KEY=your-super-secret-jwt-key-change-in-production
JWT_ALGORITHM=HS256
JWT_ACCESS_TOKEN_EXPIRE_MINUTES=30
JWT_REFRESH_TOKEN_EXPIRE_DAYS=30

# ====================================
# CINETPAY (Paiements)
# ====================================
CINETPAY_API_KEY=your-cinetpay-api-key
CINETPAY_SITE_ID=your-cinetpay-site-id
CINETPAY_SECRET_KEY=your-cinetpay-secret-key
CINETPAY_CALLBACK_URL=http://localhost:8001/cinetpay/callback
CINETPAY_NOTIFY_URL=https://yourapp.com/api/cinetpay/notify
CINETPAY_RETURN_URL=https://yourapp.com/payment/success

# ====================================
# APPLICATION
# ====================================
APP_BASE_URL=http://localhost:8001
APP_NAME=TIM CASH
ENVIRONMENT=development
DEBUG=true

# ====================================
# REDIS (Cache & Rate Limiting)
# ====================================
REDIS_URL=redis://localhost:6379/0

# ====================================
# EMAIL (Notifications)
# ====================================
SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USERNAME=your-email@gmail.com
SMTP_PASSWORD=your-app-password
MAIL_FROM=noreply@timcash.ci

# ====================================
# LOGGING
# ====================================
LOG_LEVEL=INFO

# ====================================
# ADMIN PAR DÉFAUT
# ====================================
ADMIN_EMAIL=admin@timcash.ci
ADMIN_PASSWORD=change-this-password

# ====================================
# LIMITES PAR DÉFAUT (FCFA)
# ====================================
TIM_MINI_LIMIT=500000
TIM_MAXI_LIMIT=12000000
TIM_BUSINESS_LIMIT=0

# ====================================
# TAUX DE COMMISSION (%)
# ====================================
CINETPAY_COMMISSION_RATE=0.5
WITHDRAWAL_COMMISSION_RATE=2.0
TRANSFER_COMMISSION_RATE=1.0
LOAN_INTEREST_RATE=6.0
LOAN_PENALTY_RATE=15.0

# ====================================
# STORAGE
# ====================================
UPLOAD_DIR=uploads
MAX_UPLOAD_SIZE=5242880

# ====================================
# SECURITE
# ====================================
ALLOWED_ORIGINS=http://localhost:5173,http://localhost:3000
"""
    
    with open('.env.example', 'w', encoding='utf-8') as f:
        f.write(env_example_content)
    
    print("   ✅ .env.example créé avec toutes les variables")

def create_cleanup_report():
    """Créer un rapport de nettoyage"""
    
    report = """
╔══════════════════════════════════════════════════════════════════════╗
║                                                                      ║
║               RAPPORT DE NETTOYAGE - DONNÉES HARDCODÉES              ║
║                                                                      ║
╚══════════════════════════════════════════════════════════════════════╝

✅ ACTIONS EFFECTUÉES:

1. 🔑 Clés API CinetPay
   ✅ Remplacées par variables d'environnement dans utility/transactions.py
   ⚠️  IMPORTANT: Ajoutez vos vraies clés dans le fichier .env

2. 🔗 URLs hardcodées
   ✅ Remplacées par APP_BASE_URL dans services/users.py
   ✅ Remplacées par APP_BASE_URL dans utility/send_email.py

3. 🗑️  Fichiers de données de test
   ✅ create_extended_database.py supprimé (backup créé)
   ✅ seed_data.py supprimé (backup créé)
   ✅ recreate_database.py supprimé (backup créé)
   ✅ create_mysql_extended.py supprimé (backup créé)

4. 📝 Configuration
   ✅ .env.example mis à jour avec toutes les variables
   ✅ Backups créés dans le dossier backups_hardcoded/

═══════════════════════════════════════════════════════════════════════

⚠️  ACTIONS MANUELLES REQUISES:

1. Vérifiez votre fichier .env et ajoutez:
   - CINETPAY_API_KEY=votre-vraie-clé
   - CINETPAY_SITE_ID=votre-site-id
   - CINETPAY_SECRET_KEY=votre-secret-key
   - APP_BASE_URL=https://votre-domaine.com

2. Générez une nouvelle JWT_SECRET_KEY forte:
   python -c "import secrets; print(secrets.token_urlsafe(64))"

3. Changez ADMIN_PASSWORD dans .env

4. Vérifiez main.py et main_old.py pour d'autres données hardcodées

═══════════════════════════════════════════════════════════════════════

📦 BACKUPS CRÉÉS:

Tous les fichiers modifiés ont été sauvegardés dans:
   backups_hardcoded/

Pour restaurer un fichier:
   cp backups_hardcoded/fichier.py fichier.py

═══════════════════════════════════════════════════════════════════════

🔒 SÉCURITÉ:

✅ Plus de clés API en dur dans le code
✅ Plus d'emails de test dans le code
✅ Plus de mots de passe en clair
✅ Configuration externalisée dans .env
✅ .env dans .gitignore (vérifiez!)

═══════════════════════════════════════════════════════════════════════
"""
    
    return report

if __name__ == "__main__":
    import sys
    
    print("\n🧹 NETTOYAGE DES DONNÉES HARDCODÉES - TIM CASH")
    print("=" * 70)
    
    # Demander confirmation
    print("\n⚠️  Cette opération va:")
    print("   - Supprimer les fichiers contenant des données de test")
    print("   - Remplacer les clés API par des variables d'environnement")
    print("   - Nettoyer les URLs hardcodées")
    print("   - Créer des backups de tous les fichiers modifiés")
    
    confirmation = input("\n❓ Continuer? (oui/non): ").strip().lower()
    
    if confirmation not in ['oui', 'o', 'yes', 'y']:
        print("\n❌ Opération annulée")
        sys.exit(0)
    
    print("\n🚀 Démarrage du nettoyage...")
    print("=" * 70)
    
    try:
        # 1. Nettoyer les clés API
        cleanup_cinetpay_keys()
        
        # 2. Nettoyer les URLs
        cleanup_hardcoded_urls()
        
        # 3. Supprimer les fichiers de test
        remove_test_data_files()
        
        # 4. Mettre à jour .env.example
        update_env_example()
        
        # 5. Afficher le rapport
        print("\n" + "=" * 70)
        print(create_cleanup_report())
        
        print("🎉 Nettoyage terminé avec succès!")
        print("=" * 70 + "\n")
        
        sys.exit(0)
        
    except Exception as e:
        print(f"\n❌ Erreur lors du nettoyage: {e}")
        import traceback
        traceback.print_exc()
        sys.exit(1)
