#!/usr/bin/env python3
"""
Script de migration TIM CASH
"""
import os
import sys
import subprocess
from pathlib import Path

def run_command(command, description):
    """Exécuter une commande avec gestion d'erreur"""
    print(f"\n🔄 {description}...")
    try:
        result = subprocess.run(command, shell=True, check=True, capture_output=True, text=True)
        print(f"✅ {description} - Succès")
        if result.stdout:
            print(f"   Output: {result.stdout.strip()}")
        return True
    except subprocess.CalledProcessError as e:
        print(f"❌ {description} - Erreur")
        print(f"   Error: {e.stderr}")
        return False

def install_dependencies():
    """Installer les dépendances"""
    return run_command("pip install -r requirements.txt", "Installation des dépendances")

def init_alembic():
    """Initialiser Alembic si nécessaire"""
    if not Path("alembic/versions").exists():
        print("\n📁 Création du répertoire versions...")
        Path("alembic/versions").mkdir(parents=True, exist_ok=True)
    
    # Vérifier si Alembic est déjà initialisé
    if not Path("alembic/versions").glob("*.py"):
        return run_command("alembic revision --autogenerate -m 'Initial migration'", 
                          "Création de la migration initiale")
    else:
        print("✅ Alembic déjà initialisé")
        return True

def run_migrations():
    """Exécuter les migrations"""
    return run_command("alembic upgrade head", "Exécution des migrations")

def create_admin_user():
    """Créer l'utilisateur admin par défaut"""
    print("\n👤 Création de l'utilisateur admin...")
    
    # Script Python pour créer l'admin
    admin_script = """
import sys
sys.path.append('.')
from database import SessionLocal, create_tables
from models import User, Wallet, AdminStock, TimAccountType
import bcrypt
import uuid
from datetime import datetime

# Créer les tables si elles n'existent pas
create_tables()

db = SessionLocal()

try:
    # Vérifier si l'admin existe déjà
    admin = db.query(User).filter(User.email == "admin@timcash.ci").first()
    
    if not admin:
        # Créer l'utilisateur admin
        hashed_password = bcrypt.hashpw("admin123".encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        admin = User(
            id=str(uuid.uuid4()),
            email="admin@timcash.ci",
            username="admin",
            full_name="Administrateur TIM CASH",
            phone="+225 07 00 00 00 00",
            hashed_password=hashed_password,
            is_admin=True,
            tim_account_type=TimAccountType.TIM_BUSINESS
        )
        db.add(admin)
        db.flush()
        
        # Créer le wallet admin
        admin_wallet = Wallet(
            id=str(uuid.uuid4()),
            user_id=admin.id,
            balance=12847000.0,  # 12.847M FCFA
            max_balance=None  # Illimité pour TIM BUSINESS
        )
        db.add(admin_wallet)
        
        # Créer le stock admin
        admin_stock = AdminStock(
            balance=847000000.0,  # 847M FCFA
            total_credits=0.0,
            total_debits=0.0,
            total_commissions=0.0
        )
        db.add(admin_stock)
        
        db.commit()
        print("✅ Utilisateur admin créé avec succès")
        print("   Email: admin@timcash.ci")
        print("   Mot de passe: admin123")
    else:
        print("✅ Utilisateur admin existe déjà")
        
except Exception as e:
    print(f"❌ Erreur lors de la création de l'admin: {e}")
    db.rollback()
finally:
    db.close()
"""
    
    try:
        exec(admin_script)
        return True
    except Exception as e:
        print(f"❌ Erreur lors de la création de l'admin: {e}")
        return False

def seed_sample_data():
    """Insérer des données d'exemple"""
    print("\n🌱 Insertion des données d'exemple...")
    
    seed_script = """
import sys
sys.path.append('.')
from database import SessionLocal
from models import *
import uuid
from datetime import datetime, timedelta
import json

db = SessionLocal()

try:
    # Vérifier si des données existent déjà
    if db.query(Country).count() > 0:
        print("✅ Données d'exemple déjà présentes")
        return
    
    # Créer les pays
    countries_data = [
        {
            "id": str(uuid.uuid4()),
            "name": "Côte d'Ivoire",
            "code": "CI",
            "flag": "🇨🇮",
            "total_users": 2847,
            "total_volume": 562000000.0
        },
        {
            "id": str(uuid.uuid4()),
            "name": "Sénégal",
            "code": "SN", 
            "flag": "🇸🇳",
            "total_users": 1250,
            "total_volume": 185000000.0
        }
    ]
    
    for country_data in countries_data:
        country = Country(**country_data)
        db.add(country)
        db.flush()
        
        # Ajouter des villes
        if country.code == "CI":
            cities = [
                {"name": "Abidjan", "branch_code": "TIM-CI-ABJ", "users": 1847, "volume": 450000000.0},
                {"name": "Yamoussoukro", "branch_code": "TIM-CI-YAM", "users": 500, "volume": 62000000.0},
                {"name": "Bouaké", "branch_code": "TIM-CI-BOU", "users": 500, "volume": 50000000.0}
            ]
        else:  # Sénégal
            cities = [
                {"name": "Dakar", "branch_code": "TIM-SN-DKR", "users": 950, "volume": 140000000.0},
                {"name": "Thiès", "branch_code": "TIM-SN-THI", "users": 300, "volume": 45000000.0}
            ]
        
        for city_data in cities:
            city = City(
                id=str(uuid.uuid4()),
                country_id=country.id,
                **city_data
            )
            db.add(city)
    
    # Créer des taux de commission
    commission_rates = [
        {
            "id": str(uuid.uuid4()),
            "name": "Commission CinetPay",
            "rate_type": "percentage",
            "value": 0.5,
            "category": "recharge_cinetpay",
            "description": "Commission sur les recharges CinetPay",
            "updated_by": "system"
        },
        {
            "id": str(uuid.uuid4()),
            "name": "Frais de retrait",
            "rate_type": "percentage", 
            "value": 2.0,
            "category": "withdrawal",
            "description": "Frais sur les retraits",
            "updated_by": "system"
        },
        {
            "id": str(uuid.uuid4()),
            "name": "Taux d'intérêt prêt",
            "rate_type": "percentage",
            "value": 6.0,
            "category": "loan_interest",
            "description": "Taux d'intérêt sur les prêts",
            "updated_by": "system"
        }
    ]
    
    for rate_data in commission_rates:
        rate = CommissionRate(**rate_data)
        db.add(rate)
    
    # Créer des capacités d'appareils
    devices = [
        {
            "id": str(uuid.uuid4()),
            "device_model": "Samsung Galaxy A54",
            "brand": "Samsung",
            "has_nfc": True,
            "nfc_enabled": True,
            "os_version": "Android 13",
            "user_count": 1247,
            "transaction_count": 15680,
            "success_rate": 98.7,
            "location": "Abidjan Centre",
            "country": "Côte d'Ivoire",
            "city": "Abidjan"
        },
        {
            "id": str(uuid.uuid4()),
            "device_model": "iPhone 14",
            "brand": "Apple",
            "has_nfc": True,
            "nfc_enabled": False,
            "os_version": "iOS 17.2",
            "user_count": 892,
            "transaction_count": 8920,
            "success_rate": 95.2,
            "location": "Cocody",
            "country": "Côte d'Ivoire", 
            "city": "Abidjan"
        },
        {
            "id": str(uuid.uuid4()),
            "device_model": "Tecno Spark 10",
            "brand": "Tecno",
            "has_nfc": False,
            "nfc_enabled": False,
            "os_version": "Android 12",
            "user_count": 2156,
            "transaction_count": 12450,
            "success_rate": 89.4,
            "location": "Yopougon",
            "country": "Côte d'Ivoire",
            "city": "Abidjan"
        }
    ]
    
    for device_data in devices:
        device = DeviceCapability(**device_data)
        db.add(device)
    
    db.commit()
    print("✅ Données d'exemple insérées avec succès")
    
except Exception as e:
    print(f"❌ Erreur lors de l'insertion des données: {e}")
    db.rollback()
finally:
    db.close()
"""
    
    try:
        exec(seed_script)
        return True
    except Exception as e:
        print(f"❌ Erreur lors de l'insertion des données: {e}")
        return False

def main():
    """Fonction principale de migration"""
    print("🚀 Migration TIM CASH - Initialisation de la base de données")
    print("=" * 60)
    
    # Vérifier que nous sommes dans le bon répertoire
    if not Path("requirements.txt").exists():
        print("❌ Erreur: requirements.txt non trouvé. Exécutez ce script depuis le répertoire backend/")
        sys.exit(1)
    
    steps = [
        ("Installation des dépendances", install_dependencies),
        ("Initialisation d'Alembic", init_alembic),
        ("Exécution des migrations", run_migrations),
        ("Création de l'utilisateur admin", create_admin_user),
        ("Insertion des données d'exemple", seed_sample_data)
    ]
    
    success_count = 0
    for step_name, step_func in steps:
        if step_func():
            success_count += 1
        else:
            print(f"\n❌ Échec à l'étape: {step_name}")
            break
    
    print("\n" + "=" * 60)
    if success_count == len(steps):
        print("🎉 Migration terminée avec succès!")
        print("\n📋 Informations importantes:")
        print("   • Base de données: timcash.db")
        print("   • Admin email: admin@timcash.ci")
        print("   • Admin password: admin123")
        print("   • Serveur: python -m uvicorn main:app --reload")
        print("\n⚠️  N'oubliez pas de changer le mot de passe admin en production!")
    else:
        print(f"❌ Migration échouée ({success_count}/{len(steps)} étapes réussies)")
        sys.exit(1)

if __name__ == "__main__":
    main()
