#!/usr/bin/env python3
"""
Script d'installation des dépendances Python pour TIM CASH Backend
Installe les packages un par un avec gestion d'erreurs
"""

import subprocess
import sys

def install_package(package):
    """Installe un package avec pip"""
    try:
        print(f"📦 Installation de {package}...")
        result = subprocess.run([
            sys.executable, "-m", "pip", "install", package, 
            "--only-binary=all", "--user"
        ], capture_output=True, text=True, timeout=300)
        
        if result.returncode == 0:
            print(f"✅ {package} installé avec succès")
            return True
        else:
            print(f"❌ Erreur lors de l'installation de {package}:")
            print(result.stderr)
            return False
    except subprocess.TimeoutExpired:
        print(f"⏰ Timeout lors de l'installation de {package}")
        return False
    except Exception as e:
        print(f"💥 Exception lors de l'installation de {package}: {e}")
        return False

def install_fallback_package(package, fallback_version):
    """Essaie d'installer une version de fallback"""
    try:
        print(f"🔄 Tentative avec version de fallback: {fallback_version}")
        result = subprocess.run([
            sys.executable, "-m", "pip", "install", fallback_version,
            "--only-binary=all", "--user"
        ], capture_output=True, text=True, timeout=300)
        
        if result.returncode == 0:
            print(f"✅ {fallback_version} installé avec succès")
            return True
        else:
            print(f"❌ Échec de la version de fallback")
            return False
    except Exception as e:
        print(f"💥 Exception avec fallback: {e}")
        return False

def main():
    """Installation principale"""
    print("🚀 Installation des dépendances TIM CASH Backend")
    print("=" * 50)
    
    # Packages avec versions de fallback
    packages = [
        ("fastapi==0.110.0", "fastapi==0.104.1"),
        ("uvicorn[standard]==0.27.0", "uvicorn[standard]==0.24.0"),
        ("pydantic==2.6.1", "pydantic==2.5.0"),
        ("PyJWT==2.8.0", "PyJWT==2.7.0"),
        ("requests==2.31.0", "requests==2.30.0"),
        ("python-multipart==0.0.9", "python-multipart==0.0.6"),
        ("redis==5.0.1", "redis==4.6.0"),
        ("slowapi==0.1.9", "slowapi==0.1.8"),
    ]
    
    # Packages problématiques - on les installe en dernier
    problematic_packages = [
        ("bcrypt==4.1.2", "bcrypt==4.0.1"),
        ("python-jose[cryptography]==3.3.0", "python-jose==3.3.0"),
    ]
    
    success_count = 0
    total_packages = len(packages) + len(problematic_packages)
    
    # Installation des packages normaux
    for package, fallback in packages:
        if install_package(package):
            success_count += 1
        elif install_fallback_package(package, fallback):
            success_count += 1
        else:
            print(f"⚠️  Impossible d'installer {package}")
    
    # Installation des packages problématiques
    print("\n🔧 Installation des packages nécessitant compilation...")
    for package, fallback in problematic_packages:
        if install_package(package):
            success_count += 1
        elif install_fallback_package(package, fallback):
            success_count += 1
        else:
            print(f"⚠️  Impossible d'installer {package}")
            # Essai sans cryptography pour python-jose
            if "python-jose" in package:
                if install_package("python-jose==3.3.0"):
                    success_count += 1
    
    print("\n" + "=" * 50)
    print(f"📊 Résultat: {success_count}/{total_packages} packages installés")
    
    if success_count == total_packages:
        print("🎉 Toutes les dépendances ont été installées avec succès!")
    elif success_count >= total_packages - 2:
        print("✅ Installation majoritairement réussie. Le backend devrait fonctionner.")
    else:
        print("⚠️  Plusieurs packages n'ont pas pu être installés.")
        print("💡 Essayez d'installer Rust: https://rustup.rs/")
    
    # Vérification des imports
    print("\n🧪 Test des imports...")
    try:
        import fastapi
        print("✅ FastAPI importé")
    except ImportError:
        print("❌ FastAPI non disponible")
    
    try:
        import uvicorn
        print("✅ Uvicorn importé")
    except ImportError:
        print("❌ Uvicorn non disponible")
    
    try:
        import pydantic
        print("✅ Pydantic importé")
    except ImportError:
        print("❌ Pydantic non disponible")

if __name__ == "__main__":
    main()
