from datetime import date, datetime, timedelta, timezone
import shutil
import uuid
from fastapi import File, HTTPException, UploadFile
from pydantic import BaseModel
from utility.stock import get_admin_stock
from models.models import Loan, LoanStatus, Notification, NotificationPriority, TimAccountType, Transaction, TransactionStatus, TransactionType, User
from schemas.loans import CommissionRequest, LoanRequestWithInfo, RepayLoanRequest
from services.users import get_user_wallet
from sqlalchemy.orm import Session
from utility.transactions import cinetpay_service 
from core.config import settings

async def create_loan_with_info(
    current_user: User,
    db: Session ,
    request_amount: CommissionRequest,
):
    # Vérifier les limites de prêt selon le type de compte
    user = db.query(User).filter(User.id == current_user).first()
    
    if not user:
        raise HTTPException(status_code=404, detail="Utilisateur introuvable")
    
    # if user.kyc_verifications.kyc_active == False:
    #     raise HTTPException(status_code=404, detail="Veillez un votre document pour avoir access a notre R-online")

    max_loan_amounts = {
        TimAccountType.TIM_MINI: 0,
        TimAccountType.TIM_MAXI: 10000,
        TimAccountType.TIM_BUSINESS: 10000
    }
    max_amount = max_loan_amounts.get(user.tim_account_type, 0)
    if request_amount.amount > max_amount:
        raise HTTPException(status_code=400, detail=f"Loan amount exceeds limit of {max_amount} FCFA")

    # Vérifier s'il y a déjà un prêt actif
    active_loan = db.query(Loan).filter(
        Loan.user_id == user.id,
        Loan.status.in_([LoanStatus.PENDING, LoanStatus.APPROVED, LoanStatus.ACTIVE])
    ).first()
    if active_loan:
        raise HTTPException(status_code=404, detail="You already have an active loan")

    # Vérifier le stock admin
    admin_stock = get_admin_stock(db)
    if admin_stock.balance < request_amount.amount:
        raise HTTPException(status_code=400, detail="Insufficient funds in admin stock")


    # Créer le prêt avec informations utilisateur
    interest_rate = 0.06
    total_amount = request_amount.amount * (1 + interest_rate)

    loan = Loan(
        id=str(uuid.uuid4()),
        user_id=user.id,
        amount=request_amount.amount,
        interest_rate=interest_rate,
        penalty_rate=0.15,
        remaining_balance=total_amount,
        monthly_payment=total_amount,
        total_payments=1,
        payments_completed=0,
        status=LoanStatus.APPROVED,
        due_date=datetime.utcnow() + timedelta(days=3),
        days_overdue=0,# Stocker le chemin du document verso
    )
    db.add(loan)

    # Débiter le stock admin
    admin_stock.balance -= request_amount.amount
    admin_stock.total_debits += request_amount.amount
    # Créditer le portefeuille utilisateur
    wallet = get_user_wallet(current_user, db)
    wallet.balance += request_amount.amount

    # Créer la transaction
    transaction = Transaction(
        id=str(uuid.uuid4()),
        user_id=user.id,
        transaction_type=TransactionType.PRET_MOBILE,
        amount=request_amount.amount,
        commission_amount=request_amount.amount * interest_rate,
        commission_rate=interest_rate,
        status=TransactionStatus.COMPLETED,
        description=f"Prêt SOS - Prêt d'urgence",
        reference=f"LOAN-{loan.id[:8]}",
        external_reference=loan.id
    )
    db.add(transaction)
    db.commit()

    return {
        "loan_id": loan.id,
        "amount": loan.amount,
        "interest_amount": loan.amount * interest_rate,
        "total_repayment": total_amount,
        "due_date": loan.due_date,
        "status": "approved",
    }


async def create_commission_users(
    current_user: User,
    db: Session ,
    request: CommissionRequest,
):
    user = db.query(User).filter(User.id == current_user).first()
    if not user:
        raise HTTPException(status_code=404, detail="Utilisateur introuvable")
    # Créer la transaction
    transaction = Transaction(
        id=str(uuid.uuid4()),
        user_id=user.id,
        transaction_type=TransactionType.COMMISSION,
        amount=request.amount,
        status=TransactionStatus.PENDING,
        description=f"Activations de commission Prêt d 'urgence",
        reference=f"COMM-{str(uuid.uuid4())[:8]}",

    )
    db.add(transaction)
    db.commit()
    
    payment_data = {
            "transaction_id": transaction.id,
            'currency' : "XOF",
            "amount": request.amount,  # Montant avec commission TIM CASH
            "description": transaction.description,
            "recipient_phone": user.phone , # ⚠️ Numéro associé au wallet
            "notify_url": f"{settings.CINETPAY_CALLBACK_URL}",
            "return_url": f"{settings.APP_BASE_URL}/payment/success",
            'customer_name' :user.full_name,                              
            'customer_surname' : user.last_name, 
        }

        # Appeler CinetPay
    cinetpay_response = await cinetpay_service.create_payment_url(payment_data)
        


    return {
        "transaction_id": transaction.id,
        "description": f"Paiment de commission ",
        "payment_url": cinetpay_response["payment_url"],
    }

async def repay_loan(current_user: User, db: Session):
    loan = db.query(Loan).filter(
        Loan.user_id == current_user.id,
        Loan.status.in_([LoanStatus.APPROVED, LoanStatus.ACTIVE, LoanStatus.OVERDUE])
    ).first()

    if not loan:
        raise HTTPException(status_code=404, detail="Aucun prêt actif trouvé")

    base_amount = loan.amount * (1 + loan.interest_rate)
    today = datetime.now(timezone.utc)

    # Pénalité si retard
    if today > loan.due_date:
        days_late = (today - loan.due_date).days
        loan.days_overdue = days_late
        penalty = loan.amount * loan.penalty_rate
    else:
        penalty = 0

    total_due = base_amount + penalty

    # ⚡ Paiement selon ce que l'utilisateur a dans son wallet
    wallet = get_user_wallet(current_user.id, db)
    amount_to_pay = min(wallet.balance, total_due)

    wallet.balance -= amount_to_pay
    remaining = total_due - amount_to_pay
    loan.remaining_balance = remaining

    # ✔ Mise à jour du statut du prêt
    if remaining <= 0:
        loan.status = LoanStatus.COMPLETED
    else:
        if today > loan.due_date:
            loan.status = LoanStatus.APPROVED
        else:
            loan.status = LoanStatus.APPROVED

    # Remboursement admin
    admin_stock = get_admin_stock(db)
    admin_stock.balance += amount_to_pay

    db.commit()

    # Transaction
    transaction = Transaction(
        id=str(uuid.uuid4()),
        user_id=current_user.id,
        transaction_type=TransactionType.PRET_MOBILE,
        amount=amount_to_pay,
        status=TransactionStatus.COMPLETED,
        description=f"Remboursement du prêt",
        reference=f"COMM-{str(uuid.uuid4())[:8]}",
    )
    db.add(transaction)
    db.commit()
    
    notification = Notification(
        id=str(uuid.uuid4()),
        user_id=current_user.id,
        title="Remboursement effectué",
        message=f"Votre remboursement de {amount_to_pay} XOF a été effectué avec succès.",
        type="loan_repayment",
        priority=NotificationPriority.HIGH,
        amount=amount_to_pay,
        merchant="TIM CASH",
        created_at=datetime.utcnow()
    )

    db.add(notification)
    db.commit()
    return {
        "status": "success",
        "paid_amount": amount_to_pay,
        "remaining_due": remaining,
        "penalty": penalty,
        "days_overdue": getattr(loan, "days_overdue", 0),
        "expected_total": total_due,
        "loan_status": loan.status.value
    }
